<?php

namespace App\Http\Controllers;

use App\Http\Services\Payment\PayPalService;
use App\Http\Services\Payment\StripeService;
use App\Http\Services\Payment\RazorpayService;
use App\Models\Order;
use App\Models\Artwork;
use App\Models\User;
use App\Models\Transaction;
use App\Models\Address;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Session;

class PaymentController extends Controller
{
    protected $paypalService;
    protected $stripeService;
    protected $razorpayService;

    public function __construct(
        PayPalService $paypalService,
        StripeService $stripeService,
        RazorpayService $razorpayService
    ) {
        $this->paypalService = $paypalService;
        $this->stripeService = $stripeService;
        $this->razorpayService = $razorpayService;
    }

    public function process(Request $request)
    {

        $user = Auth::user();

        $customerInfo = [
            'name' => $request->name ?? $user->fname.' '.$user->lname,
            'address' => $request->address ?? $user->address,
            'post_code' => $request->postal_code ?? $user->zip_code,
            'phone' => $request->phone ?? $user->phone,
            'email' => $request->email ?? $user->email,
            'message' => $request->notes,
            'artwork_id' => $request->artwork_id,
            'bid_amount'   => (float) ($request->bid_amount   ?? 0),
            'amount'       => (float) ($request->amount       ?? 0),
            'tax_amount'   => (float) ($request->tax_amount   ?? 0),
            'total_amount' => (float) ($request->total_amount ?? 0),
            'type' => $request->type,
            'quantity' => $request->quantity,
            'currency' => default_currency_code(),
            'payment_method' => $request->payment_method,
            'order_id' => $request->order_id ?? NULL,
        ];


        // Save customer info to session for payment success use
        Session::put('customer_info', $customerInfo);

        $gateway = $request->payment_method;


        if ($gateway === 'paypal') {
            return $this->paypalService->createOrder($customerInfo);
        }

        if ($gateway === 'stripe') {
            if (!$request->has('stripeToken')) {
                return back()->with('error', 'Stripe token missing. Please try again.');
            }
            try {
                $charge = $this->stripeService->charge($request, $customerInfo);
                return $this->handlePaymentSuccess($charge, $customerInfo, $request);
            } catch (\Exception $e) {
                return back()->with('error', $e->getMessage());
            }
        }

        if ($gateway === 'razorpay') {
            try {
                $razorpayOrderData = $this->razorpayService->createOrder($customerInfo);

                if (isset($razorpayOrderData['error'])) {
                    return response()->json(['error' => $razorpayOrderData['error']], 422);
                }

                return response()->json($razorpayOrderData, 200);

            } catch (\Throwable $e) {
                \Log::error('Razorpay Initialization Failed: ' . $e->getMessage());
                return response()->json(['error' => 'Payment initialization failed. Please try again.'], 500);
            }
        }

        if ($gateway === 'wallet') {
            if (Auth::user()->wallet_balance < $request->total_amount) {
                return redirect()->route('payment.error')->with('error',translate('Your wallet does not have sufficient balance'));
            }
            $this->handlePaymentSuccess('', $customerInfo, $request);
            Auth::user()->decrement('wallet_balance',$request->total_amount);
            return redirect()->route('thank_you')->with('success', 'Payment successful!');
        }

        return back()->with('error', translate('Invalid payment gateway'));
    }

    public function paypalSuccess(Request $request)
    {
        $result = $this->paypalService->capturePayment($request);

        if ($result['status']) {
            $request->merge(['payment_method' => 'paypal']);
            return $this->handlePaymentSuccess($result['data'], $result['customer_info'], $request);
        }

        return redirect()->route('payment.error')->withErrors($result['message'] ?? translate('Payment failed'));
    }

    public function razorpaySuccess(Request $request)
    {
        $result = $this->razorpayService->verifyPayment($request);

        if ($result['status'] == 'success') {
            return $this->handlePaymentSuccess(['id' => $result['payment_id']], $result['customer_info'], $request);
        }

        return redirect()->route('payment.error')->withErrors($result['message'] ?? 'Payment failed');
    }

    // Common handler to store order, wallet etc for all gateways
    protected function handlePaymentSuccess($paymentResponse, $customerInfo, $request)
    {
        $type = $customerInfo['type'] ?? 1;

        if (in_array($type, [2, 3])) {
            $order = new Order();
            $order->order_number = 'ORD-' . substr(uniqid(), -6);
            $order->artwork_id = $customerInfo['artwork_id'] ?? null;
            $order->user_id = Auth::id();
            $order->type = $type == 2 ? 1 : 2;
            $order->amount = $customerInfo['bid_amount'] > 0 ? $customerInfo['bid_amount'] : $customerInfo['amount'];
            $order->tax_rate = get_setting('tax_rate') ?? 0;
            $order->quantity = $customerInfo['quantity'] ?? 1;
            $order->message = $customerInfo['message'] ?? '';
            if ($type == 2 && ($customerInfo['amount'] ?? 0) > 0) {
                $order->payment_status = 1;
            } elseif ($type == 2) {
                $order->payment_status = 2;
            } elseif ($type == 3) {
                $order->payment_status = 3;
            }
            if($order->save()){
                $address = new Address;
                $address->order_id = $order->id;
                $address->name = $customerInfo['name'] ?? '';
                $address->address = $customerInfo['address'] ?? '';
                $address->post_code = $customerInfo['post_code'] ?? '';
                $address->phone = $customerInfo['phone'] ?? '';
                $address->email = $customerInfo['email'] ?? '';
                $address->save();
            }

            if ($type == 3 && isset($customerInfo['quantity'])) {
                $artwork = Artwork::findOrFail($customerInfo['artwork_id']);
                if ($artwork) $artwork->decrement('quantity', $customerInfo['quantity']);
            }
        }

        $payment = new Transaction();
        $payment->transaction_id = $paymentResponse->id ?? $paymentResponse['id'] ?? null;
        $payment->user_id = Auth::id();
        $payment->order_id = $order->id ?? $customerInfo['order_id'];
        $payment->payer_id = $paymentResponse->id ?? $paymentResponse['id'] ?? null;
        $payment->payer_email = $customerInfo['email'] ?? Auth::user()->email;
        $payment->type = $type;
        $payment->gateway_amount = $customerInfo['total_amount'] ?? 0;
        $payment->payment_details = getPaymentDetails($type, $customerInfo);
        $payment->amount = $customerInfo['amount'] ?? 0;
        $payment->tax_amount = $customerInfo['tax_amount'] ?? 0;
        $payment->total_amount = $customerInfo['total_amount'] ?? 0;
        $payment->payment_method = $customerInfo['payment_method'] ?? 'unknown';
        $payment->currency = $customerInfo['currency'] ?? 'USD';
        $payment->status = 2;
        
        // Commission calculation
        if (in_array($type, [2, 3, 7])) {
            $artwork = Artwork::findOrFail($customerInfo['artwork_id']);
            $adminCommissionRate = $artwork->users?->admin_commission ?? config('settings.merchant_commission', 10);
            $admin_commission = ($customerInfo['amount'] ?? 0) * ($adminCommissionRate / 100);
            $payment->admin_commission = $admin_commission;
            $payment->merchant_amount = ($customerInfo['amount'] ?? 0) - $admin_commission;
            $payment->admin_commission_rate = $adminCommissionRate;
            

            // Add wallet balance to merchant and admin
            if (in_array($type, [3, 7])) {
                User::find($artwork->author_id)->increment('wallet_balance', (int)$payment->merchant_amount);
            }
            
        }

        $payment->save();

        if($type == 7){
            $orders = Order::findOrFail($customerInfo['order_id']);
            $orders->payment_status = 3;
            $orders->update();
        }

        // Add wallet balance for deposit type
        if ($type == 1) {
            User::find(Auth::id())->increment('wallet_balance', $customerInfo['amount']);
            return redirect()->back()->with('success', translate('Deposit successful!'));
        }

        $artwork_type = $customerInfo['type'] == 2 ? 'auction' : 'direct';
        Session::put('artwork_type', $artwork_type);
        Session::forget('customer_info');

        return redirect()->route('thank_you')->with('success', 'Payment successful!');
    }

    public function paymentError()
    {
        return redirect()->back()->with('error',translate('Payment not completed or cancelled.'));
    }
}
