<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Location;
use Illuminate\Support\Facades\Validator;

class LocationController extends Controller
{
    /**
 * Display a listing of the resource.
 */
public function index(Request $request)
{
    $page_title = translate('Country');

    $countries = Location::whereNull('country_id')
        ->whereNull('state_id')
        ->when($request->search, function ($query) use ($request) {
            $query->where(function ($q) use ($request) {
                $q->where('name', 'LIKE', '%' . $request->search . '%')
                  ->orWhere('country_code', 'LIKE', '%' . $request->search . '%');
            });
        })
        ->orderBy('name', 'asc')
        ->paginate(10);

    return view('backend.location.index', compact('page_title', 'countries'));
}

/**
 * Store a newly created resource in storage.
 */
public function store(Request $request)
{
    $validator = Validator::make($request->all(), [
        'name' => 'required|max:255',
        'country_code' => 'required|max:255',
    ]);

    if ($validator->fails()) {
        return redirect()->back()->withErrors($validator)->withInput();
    }

    $country = new Location();
    $country->name = $request->name;
    $country->country_code = $request->country_code;
    $country->save();

    return redirect()->back()->with('success', translate('Country saved successfully'));
}

/**
 * Show the form for editing the specified resource.
 */
public function edit(Request $request, $id)
{
    $page_title = translate('Country');
    $countrySingle = Location::findOrFail($id);

    $countries = Location::whereNull('country_id')
        ->whereNull('state_id')
        ->when($request->search, function ($query) use ($request) {
            $query->where(function ($q) use ($request) {
                $q->where('name', 'LIKE', '%' . $request->search . '%')
                  ->orWhere('country_code', 'LIKE', '%' . $request->search . '%');
            });
        })
        ->orderBy('name', 'asc')
        ->paginate(10);

    return view('backend.location.country_edit', compact('page_title', 'countries', 'countrySingle'));
}

/**
 * Update the specified resource in storage.
 */
public function update(Request $request, $id)
{
    $validator = Validator::make($request->all(), [
        'name' => 'required|max:255',
        'country_code' => 'required|max:255',
    ]);

    if ($validator->fails()) {
        return redirect()->back()->withErrors($validator)->withInput();
    }

    $country = Location::findOrFail($id);
    $country->name = $request->name;
    $country->country_code = $request->country_code;
    $country->save();

    return redirect()->back()->with('success', translate('Country updated successfully'));
}

/**
 * Remove the specified resource from storage.
 */
public function destroy($id)
{
    Location::where('country_id', $id)->delete();
    Location::where('id', $id)->delete();

    return redirect()->back()->with('success', translate('Country deleted successfully'));
}

/**
 * Get states by country ID.
 */
public function get_state(Request $request)
{
    $data['states'] = Location::where('country_id', $request->country_id)
        ->whereNull('state_id')
        ->get(['name', 'id']);

    $data['option'] = translate('Select Option');

    return response()->json($data);
}

/**
 * Get cities by state ID.
 */
public function get_city(Request $request)
{
    $data['city'] = Location::whereNotNull('country_id')
        ->where('state_id', $request->state_id)
        ->get(['name', 'id']);

    $data['option'] = translate('Select Option');

    return response()->json($data);
}

/**
 * Show form to create a new state.
 */
public function state_create(Request $request, $id)
{
    $page_title = translate('Create State');
    $countrySingle = Location::findOrFail($id);

    $states = Location::where('country_id', $id)
        ->whereNull('state_id')
        ->when($request->search, function ($query) use ($request) {
            $query->where('name', 'LIKE', '%' . $request->search . '%');
        })
        ->orderBy('name', 'asc')
        ->paginate(10);

    return view('backend.location.state_create', compact('page_title', 'states', 'countrySingle'));
}

/**
 * Store a newly created state.
 */
public function state_store(Request $request, $id)
{
    $validator = Validator::make($request->all(), [
        'name' => 'required|max:255',
    ]);

    if ($validator->fails()) {
        return redirect()->back()->withErrors($validator)->withInput();
    }

    $state = new Location();
    $state->name = $request->name;
    $state->country_id = $id;
    $state->save();

    return redirect()->back()->with('success', translate('State saved successfully'));
}

/**
 * Edit an existing state.
 */
public function state_edit(Request $request, $id)
{
    $page_title = translate('State');
    $stateSingle = Location::findOrFail($id);

    $states = Location::where('country_id', $stateSingle->country_id)
        ->whereNull('state_id')
        ->when($request->search, function ($query) use ($request) {
            $query->where('name', 'LIKE', '%' . $request->search . '%');
        })
        ->orderBy('name', 'asc')
        ->paginate(10);

    return view('backend.location.state_edit', compact('page_title', 'states', 'stateSingle'));
}

/**
 * Update an existing state.
 */
public function state_update(Request $request, $id)
{
    $validator = Validator::make($request->all(), [
        'name' => 'required|max:255',
    ]);

    if ($validator->fails()) {
        return redirect()->back()->withErrors($validator)->withInput();
    }

    $state = Location::findOrFail($id);
    $state->name = $request->name;
    $state->save();

    return redirect()->back()->with('success', translate('State updated successfully'));
}

/**
 * Delete a state.
 */
public function state_destroy($id)
{
    Location::where('state_id', $id)->delete();
    Location::where('id', $id)->delete();

    return redirect()->back()->with('success', translate('State deleted successfully'));
}

/**
 * Show form to create a new city.
 */
public function city_create(Request $request, $id)
{
    $page_title = translate('Create City');
    $stateSingle = Location::findOrFail($id);

    $cities = Location::where('state_id', $id)
        ->when($request->search, function ($query) use ($request) {
            $query->where('name', 'LIKE', '%' . $request->search . '%');
        })
        ->orderBy('name', 'asc')
        ->paginate(10);

    return view('backend.location.city_create', compact('page_title', 'cities', 'stateSingle'));
}

/**
 * Store a newly created city.
 */
public function city_store(Request $request, $id)
{
    $validator = Validator::make($request->all(), [
        'name' => 'required|max:255',
    ]);

    if ($validator->fails()) {
        return redirect()->back()->withErrors($validator)->withInput();
    }

    $state = Location::findOrFail($id);

    $city = new Location();
    $city->name = $request->name;
    $city->country_id = $state->country_id;
    $city->state_id = $id;
    $city->save();

    return redirect()->back()->with('success', translate('City saved successfully'));
}

/**
 * Edit an existing city.
 */
public function city_edit(Request $request, $id)
{
    $page_title = translate('City');
    $citySingle = Location::findOrFail($id);

    $cities = Location::where('state_id', $citySingle->state_id)
        ->when($request->search, function ($query) use ($request) {
            $query->where('name', 'LIKE', '%' . $request->search . '%');
        })
        ->orderBy('name', 'asc')
        ->paginate(10);

    return view('backend.location.city_edit', compact('page_title', 'cities', 'citySingle'));
}

/**
 * Update an existing city.
 */
public function city_update(Request $request, $id)
{
    $validator = Validator::make($request->all(), [
        'name' => 'required|max:255',
    ]);

    if ($validator->fails()) {
        return redirect()->back()->withErrors($validator)->withInput();
    }

    $city = Location::findOrFail($id);
    $city->name = $request->name;
    $city->save();

    return redirect()->back()->with('success', translate('City updated successfully'));
}

/**
 * Delete a city.
 */
public function city_destroy($id)
{
    Location::where('id', $id)->delete();

    return redirect()->back()->with('success', translate('City deleted successfully'));
}

}
