<?php

namespace App\Http\Controllers\Frontend;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Page;
use Illuminate\Support\Facades\Validator;
use App\Models\Contact;
use App\Models\Artist;
use App\Models\Artwork;
use App\Models\Category;
use App\Models\Blog;
use App\Models\BlogCategory;
use App\Models\BlogComment;
use App\Models\Notify;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Auth;
use DrewM\MailChimp\MailChimp;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Session;
use Mews\Purifier\Facades\Purifier;

class HomeController extends Controller
{
    /** Show the application Home.
     *
     * @return View
     */
    public function index(Request $request)
    {
        try{
        $lang = $request->lang;
        $slug = 'home';

        $singlePageData = Page::with(['widgetContents' => function ($query) {
            $query->where('status', 1);
        }])->where('slug', '=', $slug)->first();

        if ($singlePageData) {
            $activeWidgets = $singlePageData->widgetContents;
            $title = $singlePageData->meta_title ?? $singlePageData->page_name ?? get_setting('meta_title');
            $meta_description = $singlePageData->meta_description ?? get_setting('meta_description');
            $meta_keyward = $singlePageData->getTranslation('meta_keyward') ? $singlePageData->getTranslation('meta_keyward') : get_setting('meta_keyward');
            $meta_image = get_setting('header_logo') ? url('assets/logo/' . get_setting('header_logo')) : '';

            return view('frontend.index', ['activeWidgets' => $activeWidgets, 'params' => $slug, 'title' => $title, 'meta_description' => $meta_description, 'meta_keyward' => $meta_keyward, 'meta_image' => $meta_image, 'lang' => $lang]);
        } else {
            return view('frontend.error.index');
        }
        } catch (\Throwable $th) {
            return view('frontend.error.index');
        }
    }

        /**
     * loadPagesContent
     *
     * @param  mixed $request
     * @param  string $slug
     * @return View
     */
    public function loadPagesContent(Request $request, $slug)
    {
        try{
        $lang = $request->lang;
        if ($slug == 'signup') {
            if (Auth::id()) {
                return redirect()->to('customer/dashboard');
            }
        }

        if ($slug) {
            $singlePageData = Page::where('slug', $slug)->first();
            if ($singlePageData) {
                $activeWidgets = $singlePageData->widgetContents;
                $is_bread_crumb = $singlePageData?->is_bread_crumb;
                $short_bio = $singlePageData->getTranslation('short_bio');
                $background_image = $singlePageData->background_image;
                $title = $singlePageData->getTranslation('meta_title') ?? $singlePageData->getTranslation('name') ?? get_setting('meta_title');
                $meta_description = $singlePageData->getTranslation('meta_description') ?? get_setting('meta_description');


                $meta_keyward = $singlePageData->getTranslation('meta_keyward') ? $singlePageData->getTranslation('meta_keyward') : get_setting('meta_keyward');
                $meta_image = get_setting('header_logo') ? url('assets/logo/' . get_setting('header_logo')) : '';

                if ($request->ajax()) {
                    if (isset($request->widget_name)) {
                        if ($request->widget_name == 'artworks') {
                            $widgetItem = getWidgetContent($singlePageData->id, 'artworks');
                            return $this->artworkSearch($request, $widgetItem->widget_content['per_page'], $lang);
                        }
                    }
                }

                return view('frontend.index', ['activeWidgets' => $activeWidgets, 'params' => $slug, 'title' => $title, 'meta_description' => $meta_description, 'meta_keyward' => $meta_keyward, 'meta_image' => $meta_image, 'lang' => $lang, 'is_bread_crumb' => $is_bread_crumb, 'short_bio' => $short_bio, 'background_image' => $background_image ]);
            } else {
                return view('frontend.error.index');
            }
        }
        } catch (\Throwable $th) {
            return view('frontend.error.index');
        }
    }

    /**
     * contact form submit
     *
     * @param  mixed $request
     * @return View
     */
    public function contact_submit(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|max:255',
            'email' => 'required|max:255',
            'message' => 'nullable',
            'g-recaptcha-response' => function ($attribute, $value, $fail) {
                $secretKey = get_setting('recaptcha_secret');
                $response = $value;
                $userIP = $_SERVER['REMOTE_ADDR'];
                $url = "https://www.google.com/recaptcha/api/siteverify?secret=$secretKey&response=$response&remoteip=$userIP";
                $response = Http::asForm()->post($url);
                $response = json_decode($response);
                if (!$response->success) {
                    Session::flash('g-recaptcha-response', 'Please Check Recaptcha');
                    $fail($attribute . "Google Recaptcha Failed");
                }
            }
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        $contact = new Contact;
        $contact->name = $request->name;
        $contact->email = $request->email;
        $contact->message = Purifier::clean($request->message);
        $contact->save();

        return redirect()->back()->with('success', translate('Your message has been sent successfully!'));
    }

    /**
     * Artist Detaiils
     *
     * @param  mixed $slug
     * @return View
     */
    public function artist_details(Request $request,$slug)
    {
        try{
        $lang = $request->lang;
        $artistSingle = Artist::where('slug',$slug)->first();
        $categories = Category::where('status',1)->get();
        $title = $artistSingle->getTranslation('name');
        if ($request->ajax()) {
            if (isset($request->widget_name)) {
                if ($request->widget_name == 'artist') {
                    return $this->artworkSearch($request, 12, $lang);
                }
            }
        }
       return view('frontend.template-' . selectedTheme() . '.pages.artist_details',compact('artistSingle','title','categories'));
       } catch (\Throwable $th) {
            return view('frontend.error.index');
        }
    }

    /**
     * artworkSearch
     *
     * @param  mixed  $request
     * @param  int  $item
     * @param  string  $lang
     * @return Response
     */
public function artworkSearch(Request $request, $item, $lang)
{
    $category_id = $request->category_id ?? '';
    $artist_id = $request->artist_id ?? '';
    $type = $request->art_type ?? '';
    $artwork_type = $request->artwork_type ?? '';
    $keyword = $request->keyword ?? '';
    $order_by = $request->order_by ?? 'latest';

    $currentDateTime = now();

    $artworks = Artwork::query();

    // Filter by category
    if (!empty($category_id)) {
        $categoryIds = is_array($category_id) ? $category_id : explode(',', $category_id);
        $artworks->whereHas('categories', function ($query) use ($categoryIds) {
            $query->whereIn('category_id', $categoryIds);
        });
    }

    // Filter by artist
    if (!empty($artist_id)) {
        $artistIds = is_array($artist_id) ? $artist_id : explode(',', $artist_id);
        $artworks->whereIn('artist_id', $artistIds);
    }

    // Search by keyword
    if (!empty($keyword)) {
        $artworks->where('name', 'LIKE', '%' . $keyword . '%');
    }

    // Filter sale_type
    if (!empty($type)) {
        if ($type === 'upcoming') {
            $artworks->where('sale_type', 1)
                     ->where('start_date', '>', $currentDateTime);
        } elseif ($type === 'auction') {
            $artworks->where('sale_type', 1)
                     ->where('start_date', '<=', $currentDateTime)
                     ->where('end_date', '>=', $currentDateTime);
        }
    }

    if($artwork_type){
        $artworks->where('sale_type', $artwork_type);
    }

    // Ensure categories and artwork are active
    $artworks->whereHas('categories', function ($query) {
        $query->where('status', 1);
    })->where('status', 1);

    // Apply sorting logic
    switch ($order_by) {
        case 'asc':
            $artworks->orderBy('name', 'asc');
            break;
            case 'desc':
            $artworks->orderBy('name', 'desc');
            break;
        case 'latest':
            $artworks->orderBy('created_at', 'desc');
            break;

        case 'best_selling':
            $artworks->orderBy('total_sold', 'desc');
            break;

        case 'price_low_high':
        $artworks->orderByRaw("
            COALESCE(
                CASE 
                    WHEN sale_type = 1 THEN CAST(min_bid_price AS DECIMAL(10,2))
                    WHEN sale_type = 2 AND sale_price IS NOT NULL THEN CAST(sale_price AS DECIMAL(10,2))
                    ELSE CAST(price AS DECIMAL(10,2))
                END, 0
            ) ASC
        ");
        break;

    case 'price_high_low':
        $artworks->orderByRaw("
            COALESCE(
                CASE 
                    WHEN sale_type = 1 THEN CAST(min_bid_price AS DECIMAL(10,2))
                    WHEN sale_type = 2 AND sale_price IS NOT NULL THEN CAST(sale_price AS DECIMAL(10,2))
                    ELSE CAST(price AS DECIMAL(10,2))
                END, 0
            ) DESC
        ");
        break;


        default:
            $artworks->latest();
    }

    // Pagination with query string
    $artworks = $artworks->paginate($item ?? 12)->withQueryString();

    // AJAX Response
    if ($request->ajax()) {
        $data = view('frontend.template-' . selectedTheme() . '.partials.filter_artwork', compact('artworks', 'lang'))->render();

        return response()->json([
            'status' => true,
            'artworks' => $data,
            'total' => $artworks->total(),
            'first_item' => $artworks->firstItem(),
            'last_item' => $artworks->lastItem()
        ]);
    }

    // Non-AJAX fallback
    return view('frontend.template-' . selectedTheme() . '.partials.filter_artwork', compact('artworks', 'lang'));
}

    /**
     * artwork_details
     *
     * @param  mixed $request
     * @param  string $slug
     * @return View
     */
public function artwork_details(Request $request,$slug){
        $lang = $request->lang;
    try {
        $artworkSingle = Artwork::where('slug',$slug)->first();
        $title = $artworkSingle->meta_title ?? $artworkSingle->getTranslation('name');
        $desc = strip_tags($artworkSingle->description);
        $meta_description = $artworkSingle->meta_description ?? Str::words($desc, 50, ' (...)') ?? get_setting('meta_description');
        $meta_keyward = $artworkSingle->meta_keyward ?? get_setting('meta_keyward');
        $meta_image = $artworkSingle->features_image ? uploaded_asset($artworkSingle->features_image) : uploaded_asset(get_setting('main_logo'));
        $similar_artworks = Artwork::where('sale_type',$artworkSingle->sale_type)
                            ->whereHas('users', function ($query) {
                                $query->where(function ($subQuery) {
                                    $subQuery->where('role', '!=', 1)
                                            ->orWhere(function ($q) {
                                                $q->where('role', 1)->where('isDashboard', 1);
                                            });
                                });
                            })->latest()->take(12)->get();
         return view('frontend.template-' . selectedTheme() . '.pages.artwork_details', compact('artworkSingle', 'title', 'similar_artworks', 'meta_description', 'meta_keyward', 'meta_image', 'lang'));
    } catch (\Throwable $th) {
        return view('frontend.error.index');
    }
    
}

public function notify_submit(Request $request) {
    try {
        /** Validation */
        $validator = Validator::make($request->all(), [
            'email' => 'required|max:255',
            'artwork_id' => 'required|max:255'
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        $notify = new Notify;
        $notify->email = $request->email;
        $notify->artwork_id = $request->artwork_id;
        $notify->save();
        return redirect()->back()
            ->with('success', translate('Your notification sent successfully'));
        } catch (\Throwable $th) {
            return view('frontend.error.index');
        }
}

/**
     * category page
     *
     * @param  mixed $request
     * @param  string $slug
     * @return View
     */
public function category_page(Request $request,$slug){
        $lang = $request->lang;
    try {
        $categorySingle = Category::where('slug',$slug)->first();
        $title = $categorySingle->getTranslation('name');
        $meta_description = get_setting('meta_description');
        $meta_keyward = get_setting('meta_keyward');
        $meta_image = uploaded_asset(get_setting('main_logo'));
        $artworks = $categorySingle->artworks()->latest()->paginate(12);
         return view('frontend.template-' . selectedTheme() . '.pages.category_page', compact('artworks', 'title', 'meta_description', 'meta_keyward', 'meta_image', 'lang'));
    } catch (\Throwable $th) {
        return view('frontend.error.index');
    }
    
}

/**
     * tag page
     *
     * @param  mixed $request
     * @param  string $slug
     * @return View
     */
public function tag_page(Request $request,$slug){
        $lang = $request->lang;
    try {
        $tag = str_replace('-', ' ', $slug);
        $title = $tag;
        $meta_description = get_setting('meta_description');
        $meta_keyward = get_setting('meta_keyward');
        $meta_image = uploaded_asset(get_setting('main_logo'));
        $artworks = Artwork::where('status',1)->where('tags', 'LIKE', "%{$tag}%")->latest()->paginate(12);
         return view('frontend.template-' . selectedTheme() . '.pages.tag_page', compact('artworks', 'title', 'meta_description', 'meta_keyward', 'meta_image', 'lang'));
    } catch (\Throwable $th) {
        return view('frontend.error.index');
    }
    
}

/**
     * blog details
     *
     * @param  mixed $request
     * @param  string $slug
     * @return View
     */
public function blog_details(Request $request,$slug){
        $lang = $request->lang;
    try {
        $blogSingle = Blog::where('slug',$slug)->firstOrFail();
        $previous = Blog::where('id', '<', $blogSingle->id)->orderBy('id', 'desc')->first();
        $next = Blog::where('id', '>', $blogSingle->id)->orderBy('id', 'asc')->first();
        $title = $blogSingle->meta_title ?? $blogSingle->getTranslation('name');
        $desc = strip_tags($blogSingle->description);
        $meta_description = $blogSingle->meta_description ?? Str::words($desc, 50, ' (...)') ?? get_setting('meta_description');
        $meta_keyward = $blogSingle->meta_keyward ?? get_setting('meta_keyward');
        $meta_image = $blogSingle->image ? uploaded_asset($blogSingle->image) : uploaded_asset(get_setting('main_logo'));

        $comments = BlogComment::where('blog_id', $blogSingle->id)->where('parent_id', 0)->latest()->get();
       
         return view('frontend.template-' . selectedTheme() . '.pages.blog_details', compact('blogSingle', 'previous', 'next', 'comments', 'title', 'meta_description', 'meta_keyward', 'meta_image', 'lang'));
    } catch (\Throwable $th) {
        return view('frontend.error.index');
    }
    
}

/**
     * blog category page
     *
     * @param  mixed $request
     * @param  string $slug
     * @return View
     */
public function blog_category(Request $request,$slug){
        $lang = $request->lang;
    try {
        $categorySingle = BlogCategory::where('slug',$slug)->first();
        $title = $categorySingle->getTranslation('name');
        $meta_description = get_setting('meta_description');
        $meta_keyward = get_setting('meta_keyward');
        $meta_image = uploaded_asset(get_setting('main_logo'));
        $blogs = $categorySingle->blogs()->latest()->paginate(12);
         return view('frontend.template-' . selectedTheme() . '.pages.blog_category_page', compact('blogs', 'title', 'meta_description', 'meta_keyward', 'meta_image', 'lang'));
    } catch (\Throwable $th) {
        return view('frontend.error.index');
    }
    
}

    /**
     * search page
     *
     * @param  mixed $request
     * @return View
     */
public function search(Request $request){
        $lang = $request->lang;
    try {
        $results = Artwork::where('status',1)
            ->where('name', 'like', '%' . $request->search . '%')
              ->latest()->paginate(12)->withQueryString();
        $title = 'Search: '.$request->search;
        $meta_description = get_setting('meta_description');
        $meta_keyward = get_setting('meta_keyward');
        $meta_image = uploaded_asset(get_setting('main_logo'));
         return view('frontend.template-' . selectedTheme() . '.pages.search_page', compact('results', 'title', 'meta_description', 'meta_keyward', 'meta_image', 'lang'));
    } catch (\Throwable $th) {
        return view('frontend.error.index');
    }
    
}
 /**
     * blog comment
     *
     * @param  mixed $request
     * @return View
     */
public function blog_comment(Request $request) {
    try{
        $validator = Validator::make($request->all(), [
            'name' => 'nullable|max:255',
            'email' => 'nullable|max:255',
            'comment' => 'required',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        $comments = new BlogComment;
        $comments->name = $request->name;
        $comments->email = $request->email;
        $comments->user_id = Auth::check() ? Auth::user()->id : NULL;
        $comments->blog_id = $request->blog_id;
        $comments->parent_id = $request->parent_id ? $request->parent_id : 0;
        $comments->comment = $request->comment;
        $comments->save();

        return redirect()->back()
            ->with('success', translate('Your comment sent successfully'));
        } catch (\Throwable $th) {
            return view('frontend.error.index');
        }
}

/**
     * newsletter_subscribe
     *
     * @param  mixed $request
     * @return Response
     */
    public function newsletter_subscribe(Request $request)
    {
        try {
            if($request->email == ""){
                return redirect()->back()->with('success', translate('Enter Your Email'));
            }


            $api_key = get_setting('MAILCHIMP_API_KEY');
            $list_id = get_setting('MAILCHIMP_LIST_ID');

            $MailChimp = new MailChimp($api_key);

            $result = $MailChimp->post("lists/$list_id/members", [
                'email_address' => $request->email,
                'status' => 'subscribed',
            ]);

            if ($MailChimp->success()) {
                return redirect()->back()->with('success', translate('Thanks For Subscribe'));
            } else {
                return redirect()->back()->with('error', $MailChimp->getLastError());
            }
        } catch (\Throwable $th) {
            return redirect()->back()->with('error',  translate('Credentials Wrong!'));
        }
    }

    public function certificate($slug) {

        $artworkSingle = Artwork::where('slug',$slug)->first();

        $title = translate('Certificate of Authenticity');
        $meta_description = get_setting('meta_description');
        $meta_keyward = get_setting('meta_keyward');
        $meta_image = get_setting('header_logo') ? url('assets/logo/' . get_setting('header_logo')) : '';

        return view('frontend.template-' . selectedTheme() . '.pages.certificate', compact('artworkSingle', 'title', 'meta_description', 'meta_keyward', 'meta_image'));
    }
}
