<?php

namespace App\Http\Controllers\Backend;

use App\Http\Controllers\Controller;
use App\Http\Requests\ProfileUpdateRequest;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Redirect;
use Illuminate\View\View;
use App\Models\Location;
use App\Models\Artist;
use App\Models\ArtGalleryPaymentInfo;
use Illuminate\Support\Facades\Hash;

class ProfileController extends Controller
{
    /**
     * Display the user's profile form.
     */
    public function edit(Request $request): View
    {
        $page_title = translate('Profile Update');
        $user = Auth::user();
        if($user->role == 1){
            if(!$user->artists){
                return redirect()->back()->with('error', translate('Artist not found'));
            }
            $lang = $request->lang ?? 'en';
            $page_title = translate('Edit Artist');
            $artistSingle = Artist::findOrFail($user->artists?->id);

            return view('backend.artists.edit', compact('artistSingle', 'page_title', 'lang'));
        }else{
            $countries = Location::whereNull('country_id')->whereNull('state_id')->orderBy('name','asc')->get();
            return view('backend.dashboard.profile', compact('page_title', 'user','countries'));
        }
    }

    /**
     * Update the user's profile information.
     */
    public function update(ProfileUpdateRequest $request): RedirectResponse
    {
        $user = $request->user();
        $data = $request->validated();
        if (!empty($data['password'])) {
            $data['password'] = Hash::make($data['password']);
        }else {
            // Remove password key so it won't overwrite existing password with NULL
            unset($data['password']);
        }
        if ($request->hasFile('image')) {
            $image = $request->file('image');
            // Delete old image if exists
            if ($user->image && file_exists(file_path() . $user->image)) {
                @unlink(file_path() . $user->image);
            }

            $data['image'] = uploadImage($image);
        }

        $user->fill($data);
        $user->save();

        return Redirect::route('dashboard.profile.edit')->with('success', translate('Profile Updated Successfully'));
    }

     /**
     * Show the form for editing the specified resource.
     */
    public function payment()
    {
        $page_title = translate('Payment Info');
        $user = Auth::user();
        $art_gallery_payment = ArtGalleryPaymentInfo::where('user_id', $user->id)->get();

        return view('backend.artists.payment_info', compact('page_title', 'art_gallery_payment', 'user'));
    }

    /**
     * Update the user's payment information.
     */
    public function payment_update(Request $request, $id)
    {
        $this->savePaymentInfo($id, $request);

        return redirect()->back()->with('success', translate('Payment Information has been updated successfully'));
    }

    private function savePaymentInfo($user_id, Request $request)
{
    if (!$request->payment_type) return;
    // Delete all previous payment info for this user
    ArtGalleryPaymentInfo::where('user_id', $user_id)->delete();

    foreach ($request->payment_type as $key => $type) {

        $options = match ((int)$type) {
            1 => [
                'bank_name'           => $request->bank_name[$key] ?? null,
                'branch_name'         => $request->branch_name[$key] ?? null,
                'bank_ac_name'        => $request->bank_ac_name[$key] ?? null,
                'bank_ac_number'      => $request->bank_ac_number[$key] ?? null,
                'bank_routing_number' => $request->bank_routing_number[$key] ?? null,
            ],
            2 => [
                'mobile_banking_name'   => $request->mobile_banking_name[$key] ?? null,
                'mobile_banking_number' => $request->mobile_banking_number[$key] ?? null,
            ],
            3 => [ 
                'paypal_name'          => $request->paypal_name[$key] ?? null,
                'paypal_username'      => $request->paypal_username[$key] ?? null,
                'paypal_email'         => $request->paypal_email[$key] ?? null,
                'paypal_mobile_number' => $request->paypal_mobile_number[$key] ?? null,
            ],
            default => [],
        };

        ArtGalleryPaymentInfo::create([
            'user_id'       => $user_id,
            'payment_type'  => $type,
            'payment_info'  => json_encode($options),
        ]);
    }
}

}
