<?php

namespace App\Http\Controllers\Backend;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Page;
use App\Models\PageTranslation;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use App\Models\Widget;
use App\Models\WidgetContent;
use App\Models\WidgetContentTranslation;
use Mews\Purifier\Facades\Purifier;
use Illuminate\Support\Facades\File;

class PageController extends Controller
{
    public function __construct()
    {
        $this->middleware(['auth', 'admin']);
    }
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $pages = Page::orderBy('id', 'asc')->paginate(15);
        $page_title = translate('Page List');
        return view('backend.pages.index', compact('pages', 'page_title'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        /** Validation */
        $validator = Validator::make($request->all(), [
            'name' => 'required|max:255',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        $pages = new Page;
        $pages->name = $request->name;
        $slug = Str::slug($request->name, '-');
        $same_slug_count = Page::where('slug', 'LIKE', $slug . '%')->count();
        $slug_suffix = $same_slug_count ? '-' . $same_slug_count + 1 : '';
        $slug .= $slug_suffix;

        $pages->slug = $slug;
        $pages->save();

        return redirect()->back()->with('success', translate('Page saved successfully'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Request $request, $id)
    {
        $page = Page::with('widgetContents.widget', 'widgetContents.widgetTranslations')->find($id);
        $page_title = $page->getTranslation('name');
        $lang = $request->lang;
        $widgetList = Widget::orderBy('widget_name','asc')->get();
        return view('backend.pages.edit', compact('page', 'widgetList', 'page_title', 'lang'));
    }

     /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        try {

            $validator = Validator::make($request->all(), [
                'name' => 'required|max:255',
                'slug' => 'required|unique:pages,slug,' . $request->id,
            ]);

            if ($validator->fails()) {
                return redirect()->back()->withErrors($validator)->withInput();
            }

            $page = Page::findOrFail($id);

            if ($meta_image = $request->file('meta_image')) {
                if ($page->meta_image && file_exists(file_path().$page->meta_image)) {
                    unlink(file_path().$page->meta_image);
                }
                $page->meta_image = uploadImage($meta_image);
            }

            if ($background_image = $request->file('background_image')) {
                if ($page->background_image && file_exists(file_path().$page->background_image)) {
                    unlink(file_path().$page->background_image);
                }
                $page->background_image = uploadImage($background_image);
            }



            if ($request->lang ==  get_setting("DEFAULT_LANGUAGE", "en") ||  $request->lang == "") {
                $page->name = prelaceScript(html_entity_decode($request->name));
                $page->slug = prelaceScript(html_entity_decode($request->slug));
                $page->meta_title = $request->meta_title;
                $page->meta_description = str_replace('script', '', prelaceScript(html_entity_decode($request->meta_description)));
                $page->meta_keyward = $request->meta_keyward;
                $page->enable_seo = $request->enable_seo == "on" ? 1 : 2;
                $page->is_bread_crumb = $request->is_bread_crumb == "on" ? 1 : 2;
                $page->short_bio = str_replace('script', '', prelaceScript(html_entity_decode($request->short_bio)));
                $page->update();
            } else {
                $page_translation = PageTranslation::firstOrNew(['lang' => $request->lang, 'page_id' => $page->id]);
                $page_translation->name = prelaceScript(html_entity_decode($request->name));
                $page_translation->meta_title = $request->meta_title;
                $page_translation->meta_description = str_replace('script', '', prelaceScript(html_entity_decode($request->meta_description)));
                $page_translation->meta_keyward = $request->meta_keyward;
                $page_translation->short_bio = str_replace('script', '', prelaceScript(html_entity_decode($request->short_bio)));
                $page_translation->save();
            }


            return  response()->json(['status' => true, 'message' => 'Update Successfully']);
        } catch (\Throwable $th) {
            //throw $th;
            return  response()->json(['status' => false, 'message' => $th->getMessage()]);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        $pages = Page::with('widgetContents')->findOrFail($id);

        if ($pages->delete()) {
            $pages->widgetContents()->delete();
        }
        return back()->with('success', translate('Page deleted successfully'));
    }

    /**
     * changeStatus
     *
     * @return Response
     */
    public function changeStatus()
    {
        $status         = $_POST['status'];
        $pageId     = $_POST['pageId'];

        if ($status && $pageId) {
            $pages = Page::findOrFail($pageId);
            if ($status == 1) {
                $pages->status = 2;
                $message = translate('Page Inactive');
            } else {
                $pages->status = 1;
                $message = translate('Page Active');
            }
            if ($pages->update()) {
                $response = array('output' => 'success', 'statusId' => $pages->status, 'pageId' => $pages->id, 'message' => $message, 'active' => translate('Active'), 'inactive' => translate('Inactive'));
                return response()->json($response);
            }
        }
    }

    /**
     * getWidgetContent
     *
     * @return View
     */
    public function getWidgetContent()
    {

        $posts              = $_REQUEST;
        $random             = substr(md5(mt_rand()), 0, 7);
        $singlePageData     = Page::find($_REQUEST['pageId']);

        /*---Page Content Update------*/
        $widgetContentUpdate = $random;
        if ($singlePageData->widget_content_code) {
            $widgetContentUpdate = $singlePageData->widget_content_code . ',' . $random;
        }
        $singlePageData->widget_content_code = $widgetContentUpdate;
        $singlePageData->update();

        /*---Create content card------*/
        $singleWidgetContentModel = new WidgetContent;
        $singleWidgetContentModel->page_id       = $posts['pageId'];
        $singleWidgetContentModel->widget_slug   = $posts['slug'];
        $singleWidgetContentModel->ui_card_number = $random;
        $singleWidgetContentModel->save();

        return view('backend.pages.widgets.' . $_REQUEST['slug'] . '', ['posts' => $posts, 'randomId' => $random]);
    }


    /** =========== widget added to page ===========
     * =========== widgetAddedToPage ===========
     *
     * @param string slug (widget_slug)
     * @return Response
     */

    public function widgetAddedToPage($slug)
    {

        $pageId = Request()->pageId;
        $widgetName = Request()->widgetName;
        $randomId = substr(md5(mt_rand()), 0, 7);;
        $content = view('backend.pages.widgets.' . $slug, compact('randomId', 'pageId', 'slug', 'widgetName'))->render();
        $this->storeWidgetByPage($pageId, $randomId, $slug);
        return  response()->json(['content' => $content, 'pageId' => $pageId, 'status' => true, 'message' => 'Add Successfully']);
    }


    /** widget update by page
     * ===========  widgetUpdateByPage =============
     *
     * @param mix Request
     * @return Response
     */
    public function widgetUpdateByPage(Request $request)
    {

        try {

            $widgetsContent = WidgetContent::where('ui_card_number', $request->ui_card_number)->where('widget_slug', $request->widget_slug)->first();

            if ($request->lang ==  get_setting("DEFAULT_LANGUAGE", "en") ||  $request->lang == "") {

                $widgetsContent->ui_card_number = $request->ui_card_number;
                $widgetsContent->widget_slug = $request->widget_slug;
                $widgetsContent->page_id = $request->page_id;
                $widgetsContent->widget_content = isset($request->content[0]) ? Purifier::clean( $request->content[0] , array('Attr.EnableID' => true)) : null;
                $widgetsContent->update();
            } else {
                $this->widgetTranslate($widgetsContent->id, $request);
            }



            return  response()->json(['status' => true, 'message' => 'Update Successfully']);
        } catch (\Throwable $th) {
            return  response()->json(['status' => false, 'message' => $th->getMessage()]);
        }
    }


    /** status change
     * ======= statusChange ========
     *
     * @param  int id
     * @return Response
     */

    public function widgetStatusChange($id)
    {
        try {
            $result = WidgetContent::where('id', $id)->orWhere('ui_card_number', $id)->first();
            $result->status = $result->status == 1 ? 0 : 1;
            $result->update();
            return  response()->json(['status' => true,  'message' => translate('Status Change Successfully')]);
        } catch (\Throwable $th) {

            return  response()->json(['status' => false, 'message' => translate('Something Wrong!')]);
        }
    }

    /** widget delete by page
     * ======= statusChange ========
     *
     * @param  int id
     * @return Response
     */

    public function widgetDeleteByPage($id)
    {
        try {
            $result = WidgetContent::where('id', $id)->orWhere('ui_card_number', $id)->first();
            if ($result->delete()) {
                return  response()->json(['status' => true,  'message' => translate('Widget Delete Successfully')]);
            }
        } catch (\Throwable $th) {
            return  response()->json(['status' => false, 'message' => translate('Something Wrong!')]);
        }
    }


    /** sorted widget
     *======== widgetSortedByPage ==============
     *
     * @param Request
     * @return Response
     */

    public function widgetSortedByPage(Request $request)
    {

        try {
            if (isset($request->content)) {
                $count = 0;
                foreach ($request->content as  $slug) {
                    $key = key($slug);
                    $value = $slug[$key];
                    $count++;
                    if ($widgetsContent = WidgetContent::where('ui_card_number', $key)->where('widget_slug', $value)->first()) {
                        $widgetsContent->position = $count;
                        $widgetsContent->update();
                    } else {
                        $this->storeWidgetByPage($request->pageId, $key, $value);
                    }
                }
                return  response()->json(['status' => true,  'message' => translate('Update Successfully')]);
            }
        } catch (\Throwable $th) {
            return  response()->json(['status' => false, 'message' => translate('Something Wrong!')]);
        }
    }

    /** store widget to page
     * =========== storeWidgetByPage ============
     *
     * @param int PageId
     * @param string randomId
     * @param  string slug {widget slug}
     * @return Response
     */
    public function storeWidgetByPage($pageId, $randomId, $slug)
    {

        $widget =  WidgetContent::where("page_id", $pageId)->select('position')->orderBy('position', 'DESC')->latest()->first();
        $widgetsContent =  new  WidgetContent();
        $widgetsContent->ui_card_number = $randomId;
        $widgetsContent->widget_slug = $slug;
        $widgetsContent->page_id = $pageId;
        $widgetsContent->position = $widget ? $widget->position += 1 : 1;
        $widgetsContent->save();
    }



    /** base64 image upload
     * ============ imageUpload ========
     *
     * @param Request
     * @return Response
     */

    public function imageUpload(Request $request)
    {

        try {
            $fileName = $this->base64ImgUpload($request->image, $file = $request->old_file ? $request->old_file : "", $folder = $request->folder);
            return response()->json(['status' => true, 'image_name' => $fileName]);
        } catch (\Throwable $th) {
            return response()->json(['status' => false, 'message' => $th->getMessage()]);
        }
    }


    /**
     * base64ImgUpload
     *
     * @param  mixed $requesFile
     * @param  mixed $file
     * @param  mixed $folder
     * @return Response
     */
public static function base64ImgUpload($requesFile, $file, $folder)
{
    // Ensure it's a base64 image
    if (!str_starts_with($requesFile, 'data:image')) {
        throw new \Exception('Invalid image input.');
    }

    // Split metadata and data
    [$meta, $content] = explode(',', $requesFile, 2);

    // Detect extension
    $extension = 'png'; // default
    if (preg_match('/^data:image\/([a-zA-Z0-9\+\-\.]+);base64$/', $meta, $matches)) {
        $extension = strtolower($matches[1]);
        if ($extension === 'jpeg') $extension = 'jpg';
        if ($extension === 'svg+xml') $extension = 'svg';
    }

    $imageName = "egens-" . \Illuminate\Support\Str::random(10) . '.' . $extension;

    // Delete old file if exists
    if (!empty($file) && file_exists(public_path($folder . $file))) {
        unlink(public_path($folder . $file));
    }

    // Ensure folder exists
    if (!file_exists(public_path($folder))) {
        mkdir(public_path($folder), 0755, true);
    }

    // Save file
    $decoded = base64_decode($content);
    if ($decoded === false) {
        throw new \Exception('Base64 decode failed.');
    }

    file_put_contents(public_path($folder . $imageName), $decoded);

    return $imageName;
}








    /**  Page tranlsate
     *
     *============ translate ============
     *
     * @param request
     * @return response
     */

    public function translate($request)
    {
        return PageTranslation::updateOrCreate(['page_id' => $request->id, 'lang' => $request->lang], [
            'name' => prelaceScript(html_entity_decode($request->page_name)),
            'slug' => prelaceScript(html_entity_decode( $request->page_slug)),
            'meta_title' => $request->meta_title,
            'meta_description' => str_replace('script', '', prelaceScript(html_entity_decode($request->meta_description))),
            'meta_keyward' => $request->meta_keyward,
            'short_bio' => str_replace('script', '', prelaceScript(html_entity_decode($request->short_bio))),

        ]);
    }


    /**  widget tranlsate
     *
     *============ translate ============
     *
     * @param request
     * @return response
     */

    public function widgetTranslate($id, $request)
    {
        return WidgetContentTranslation::updateOrCreate(['widget_content_id' => $id, 'lang' => $request->lang], [
            'page_id' => $request->page_id,
            'widget_content_id' => $id,
            'widget_content' => isset($request->content[0]) ? $request->content[0] : null,
        ]);
    }
}
