<?php

namespace App\Http\Controllers\Backend;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Auth;
use App\Models\User;
use App\Models\Artwork;
use App\Models\Order;
use App\Models\Transaction;
use Illuminate\Support\Facades\DB;

class HomeController extends Controller
{
    public function index()
    {
        $page_title = translate('Dashboard');
        $user = Auth::user();
        $isAdmin = $user->role == 3;

        $user_sale_artworks    = Artwork::where('sale_type', 2)->when(!$isAdmin, fn($q) => $q->where('author_id', $user->id))->pluck('id');
        $user_bidding_artworks = Artwork::where('sale_type', 1)->when(!$isAdmin, fn($q) => $q->where('author_id', $user->id))->pluck('id');
        $user_all_artworks = Artwork::when(!$isAdmin, fn($q) => $q->where('author_id', $user->id))->pluck('id');

        $data['total_auction'] = Artwork::where('sale_type', 1)
            ->when(!$isAdmin, fn($q) => $q->where('author_id', $user->id))->count();

        $data['total_direct'] = Artwork::where('sale_type', 2)
            ->when(!$isAdmin, fn($q) => $q->where('author_id', $user->id))->count();

        $data['total_bid'] = Order::where('type', 1)
            ->when(!$isAdmin, fn($q) => $q->where('user_id', $user->id))->count();

        $data['total_order'] = Order::where('type', 2)
            ->when(!$isAdmin, fn($q) => $q->where('user_id', $user->id))->count();

        $data['pending_bid'] = Order::where('type', 1)->where('status',1)
            ->when(!$isAdmin, fn($q) => $q->where('user_id', $user->id))->count();

        $data['pending_order'] = Order::where('type', 2)->where('status',1)
            ->when(!$isAdmin, fn($q) => $q->where('user_id', $user->id))->count();

        $data['total_deposit'] = Transaction::where('type', 1)->where('status', 2)
            ->when(!$isAdmin, fn($q) => $q->where('user_id', $user->id))->sum('total_amount');

        $data['total_withdraw'] = Transaction::where('type', 4)->where('status', 2)
            ->when(!$isAdmin, fn($q) => $q->where('user_id', $user->id))->sum('total_amount');

        $data['purchaseOrderReports'] = $this->orderTypeSales($orderType = 2, $orderStatus = 3,  $user_sale_artworks);
        $data['bidOrderReports'] = $this->orderTypeSales($orderType = 1, $orderStatus = 3, $user_bidding_artworks);
        $data['orderSummeries'] = $this->orderSummeryReport($user_all_artworks);
        $data['total_income'] = Transaction::whereIn('type', [2, 3, 7])->where('status', 2)
            ->when(!$isAdmin, fn($q) => $q->where('user_id', $user->id))->where('user_id', '!=', $user->id)
        ->sum($isAdmin ? 'admin_commission' : 'merchant_amount');

        if ($isAdmin) {
            $data['total_customer'] = User::where('role', 1)->count();
            $data['total_artgallery'] = User::where('role', 2)->count();
            $data['total_tax'] = Transaction::whereIn('type', [2, 3, 7])->where('status', 2)->sum('tax_amount');
            $data['depositReports'] = $this->transitionReport(2, 1);
            $data['widthdrawReports'] = $this->transitionReport(2, 4);
            $data['customers'] = $this->userSummeryReport(1, 1);
            $data['artgallery'] = $this->userSummeryReport(2, 1);
        }
        return view('backend.dashboard.index',compact('page_title','data'));
    }

    public function cacheClear()
    {
        Artisan::call('optimize:clear');
        return back()->with('success', 'Cache Clear Successfully');
    }

     /**
     * transitionReport
     *
     * @param  mixed $status
     * @return Response
     */
   public function transitionReport($status, $type, $user_id = null)
{
    $query = Transaction::select(
        DB::raw('SUM(amount) as total_amount'),
        DB::raw("DATE_FORMAT(created_at, '%M %Y') as monthsYears")
    )
    ->where('status', $status)
    ->where('type', $type);

    if ($user_id) {
        $query->where('user_id', $user_id);
    }

    return $query->groupBy('monthsYears')->get();
}

 /**
     * userSummeryReport
     *
     * @param  int $type
     * @param  int $status
     * @return Response
     */
    public function userSummeryReport($type = null, $status = null)
    {
        return User::select(
            DB::raw('COUNT(*) as total'),
            DB::raw("DATE_FORMAT(created_at,'%d %M %Y') as monthsYears")
        )
            ->where('role', $type)
            ->where('status', $status)
            ->groupBy('monthsYears')
            ->get();
    }

     /**
     * orderTypeSales
     *
     * @param  mixed $artwork
     * @param  int   $orderType
     * @param  int   $orderStatus
     * @return Response
     */

    public function orderTypeSales($orderType, $orderStatus, $artwork = null)
    {

        $order = Order::select(
            DB::raw('SUM(quantity) as quantity'),
            DB::raw('SUM(amount) as amount'),
            DB::raw("DATE_FORMAT(created_at,'%d %M %Y') as monthsYears")
        )
            ->where('type', $orderType)
            ->where('payment_status', $orderStatus);

        if (!empty($artwork)) {
            return $order->whereIn('artwork_id', $artwork)
                ->groupBy('monthsYears')
                ->get();
        } else {
            return $order
                ->groupBy('monthsYears')
                ->get();
        }
    }

    /**
     * orderSummeryReport
     *
     * @param  Object $artwork
     * @return Response
     */
    public function orderSummeryReport($artwork = null)
    {

        $order = Order::select(
            DB::raw('SUM(quantity) as quantity'),
            DB::raw("DATE_FORMAT(created_at,'%d %M %Y') as monthsYears")
        );

        if (!empty($artwork)) {
            return $order->whereIn('artwork_id', $artwork)
                ->groupBy('monthsYears')
                ->get();
        } else {
            return $order
                ->groupBy('monthsYears')
                ->get();
        }
    }
}
