<?php

namespace App\Http\Controllers\Backend;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Category;
use Illuminate\Support\Str;
use App\Models\CategoryTranslation;
use Illuminate\Support\Facades\Validator;

class CategoryController extends Controller
{
    public function __construct()
    {
        $this->middleware(['auth', 'admin']);
    }
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $page_title = translate('Artwork Categories');
        $categories = Category::latest()->paginate(10);
        return view('backend.artwork.category.index', compact('page_title', 'categories'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        /** Validation */
        $validator = Validator::make($request->all(), [
            'name' => 'required|max:255',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }


        $category = new Category;

        $category->name = $request->name;
        $slug = Str::slug($request->name, '-');
        $same_slug_count = Category::where('slug', 'LIKE', $slug . '%')->count();
        $slug_suffix = $same_slug_count ? '-' . $same_slug_count + 1 : '';
        $slug .= $slug_suffix;

        $category->slug = $slug;
        $category->save();
        return redirect()->back()->with('success', translate('Category saved successfully'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Request $request, $id)
    {
        $page_title = translate('Edit Category');
        $lang = $request->lang;
        $categorySingle = Category::findOrFail($id);
        return view('backend.artwork.category.edit', compact('page_title', 'categorySingle', 'lang'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
         /** Validation */
        $validator = Validator::make($request->all(), [
            'name' => 'required|max:255',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }


        $category = Category::findOrFail($id);
        if ($request->lang == get_setting("DEFAULT_LANGUAGE", "en")) {
            $category->name = $request->name;
        }

        if($category->update()){
            $category_translation = CategoryTranslation::firstOrNew(['lang' => $request->lang, 'category_id' => $category->id]);
            $category_translation->name = $request->name;
            $category_translation->save();
        }
        return redirect()->back()->with('success', translate('Category has been updated successfully'));
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $category = BlogCategory::findOrFail($id);
        $category->delete();
        return back()->with('success', translate('Category deleted successfully'));
    }

     /**
     * Change Category status.
     */

    public function changeStatus(Request $request)
    {
        $data_id     = $request->data_id;
        if ($data_id) {
            $category = Category::findOrFail($data_id);
            if ($category->status == 1) {
                $category->status = 2;
                $message = translate('Category Inactive');
            } else {
                $category->status = 1;
                $message = translate('Category Active');
            }
            if ($category->update()) {
                $response = array('output' => 'success', 'statusId' => $category->status, 'dataId' => $category->id, 'message' => $message);
                return response()->json($response);
            }
        }
    }
}
