<?php

namespace App\Http\Controllers\Backend;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Blog;
use Illuminate\Support\Str;
use App\Models\BlogCategory;
use App\Models\BlogTranslation;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Mews\Purifier\Facades\Purifier;

class BlogController extends Controller
{
    public function __construct()
    {
        $this->middleware(['auth', 'admin']);
    }
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $page_title = translate('Blog List');

        $blogs = Blog::query()
        ->when($request->search, function ($query) use ($request) {
            $query->where('title', 'like', '%' . $request->search . '%');
        })->latest()->paginate(10);

        return view('backend.blog.index', compact('page_title', 'blogs'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $page_title = translate('Create Blog');
        $categories = BlogCategory::where('status', 1)->orderBy('name', 'asc')->get();
        return view('backend.blog.create', compact('categories', 'page_title'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $user = Auth::user();
        /** Validation */
        $validator = Validator::make($request->all(), [
            'title' => 'required|max:255|unique:blogs,title',
            'description' => 'required',
            'image' => 'required|mimes:jpeg,png,jpg,gif,svg,webp',
            'category_id' => 'required|max:255',
            'tags' => 'nullable',
        ]);


        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        $blogs = new Blog;
        if ($image = $request->file('image')) {
            $blogs->image = uploadImage($image);
        }

        if ($meta_image = $request->file('meta_image')) {
            $blogs->meta_image = uploadImage($meta_image);
        }

        $blogs->title = $request->title;
        $blogs->description = Purifier::clean($request->description);
        $blogs->tags = $request->tags;
        $blogs->category_id = $request->category_id;
        $slug = Str::slug($request->title, '-');
        $same_slug_count = Blog::where('slug', 'LIKE', $slug . '%')->count();
        $slug_suffix = $same_slug_count ? '-' . $same_slug_count + 1 : '';
        $slug .= $slug_suffix;
        $blogs->slug = $slug;
        $blogs->enable_seo = $request->enable_seo == "on" ? 1 : null;

        $blogs->meta_title = $request->meta_title;
        $blogs->meta_keyward = $request->meta_keyward;
        $blogs->meta_description = Purifier::clean($request->meta_description);
        $blogs->user_id = $user->id;

        if ($user->role == 2) {
            $blogs->status = 2;
        }else{
            $blogs->status = $request->status;
        }

        $blogs->save();

        return redirect()->route('dashboard.blogs.index')->with('success', translate('Blog saved successfully'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Request $request, $id)
    {
        $page_title = translate('Edit Blog');
        $lang = $request->lang;
        $blogSingle = Blog::findOrFail($id);
        $categories = BlogCategory::where('status', 1)->orderBy('name', 'asc')->get();
        return view('backend.blog.edit', compact('page_title', 'categories', 'blogSingle', 'lang'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {

        $blogs = Blog::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'title' => 'required|max:255|unique:blogs,title,'.$id,
            'description' => 'required',
            'category_id' => 'required|max:255',
            'tags' => 'nullable',
        ]);

        if($request->hasFile('image')){
            $validator = Validator::make($request->all(), [
                'image' => 'required|mimes:jpeg,png,jpg,gif,svg,webp',
            ]);
         }
        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        if ($request->lang == get_setting("DEFAULT_LANGUAGE", "en")) {
            $blogs->title = $request->title;
            $blogs->description = Purifier::clean( $request->description);
            $blogs->tags = $request->tags;
        }

        if ($image = $request->file('image')) {
            if ($blogs->image && file_exists(file_path().$blogs->image)) {
                unlink(file_path().$blogs->image);
            }
            $blogs->image = uploadImage($image);
        }

        if ($meta_image = $request->file('meta_image')) {
            if (file_exists(file_path().$blogs->meta_image)) {
                unlink(file_path().$blogs->meta_image);
            }
            $blogs->meta_image = uploadImage($meta_image);
        }

        $blogs->category_id = $request->category_id;
        $blogs->meta_title = $request->meta_title;
        $blogs->meta_keyward = $request->meta_keyward;
        $blogs->meta_description = Purifier::clean($request->meta_description);

        $blogs->enable_seo = $request->enable_seo == "on" ? 1 : null;
        $blogs->status = $request->status;
        $blogs->update();

        $blog_translation = BlogTranslation::firstOrNew(['lang' => $request->lang, 'blog_id' => $blogs->id]);
        $blog_translation->title = $request->title;
        $blog_translation->description = Purifier::clean($request->description);
        $blog_translation->tags = $request->tags;
        $blog_translation->save();

        return redirect()->route('dashboard.blogs.index')->with('success', translate('Blog has been updated successfully'));
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        $blogs = Blog::findOrFail($id);
        if ($blogs->image && file_exists(file_path().$blogs->image)) {
            unlink(file_path() . $blogs->image);
        }
        if ($blogs->meta_image && file_exists(file_path().$blogs->meta_image)) {
            unlink(file_path() . $blogs->meta_image);
        }
        
        $blogs->delete();
        return back()->with('success', translate('Blog deleted successfully'));
    }

    /**
     * Change Blog status.
     */

    public function changeStatus(Request $request)
    {
        $data_id     = $request->data_id;
        if ($data_id) {
            $blogs = Blog::findOrFail($data_id);
            if ($blogs->status == 1) {
                $blogs->status = 2;
                $message = translate('Blog Inactive');
            } else {
                $blogs->status = 1;
                $message = translate('Blog Active');
            }
            if ($blogs->update()) {
                $response = array('output' => 'success', 'statusId' => $blogs->status, 'dataId' => $blogs->id, 'message' => $message);
                return response()->json($response);
            }
        }
    }
}
