<?php

namespace App\Http\Controllers\Backend;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Currency;
use App\Models\Setting;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Mail;

class BackendSettingController extends Controller
{
    public function __construct()
    {
        $this->middleware(['auth', 'admin']);
    }
    
    public function index()
    {
        return view('backend.backend_setting.index', [
            'page_title' => translate('Backend Settings'),
            'currencies' => Currency::where('status', 1)->get(),
            'backend_setting' => Setting::all(),
        ]);
    }

public function store(Request $request)
{
    $defaultOffSettings = [
        'paypal_sandbox_mode', 'tawk_enabled', 'google_recapcha_check',
        'paypal_status', 'stripe_status', 'razorpay_status',
        'artgallry_email_verification', 'customer_email_verification',
    ];

    // First update all toggle-type settings
    foreach ($defaultOffSettings as $key) {
        Setting::updateOrCreate(
            ['type' => $key],
            ['value' => $request->has($key) ? 1 : 0]
        );
    }

    // Now handle the rest
    foreach ($request->except('_token') as $key => $value) {
        $setting = Setting::firstOrNew(['type' => $key]);

        switch ($key) {
            case 'started_date':
                $setting->value = date('d-m-Y', strtotime($value));
                break;

            case 'admin_logo':
            case 'invoice_logo':
            case 'authorized_sign':
                if ($request->hasFile($key)) {
                    $old = get_setting($key);
                    if ($old && file_exists(file_path().$old)) {
                        @unlink(file_path().$old);
                    }
                    $setting->value = uploadImage($request->file($key));
                }
                break;

            case 'company_name':
                $this->overWriteEnvFile('APP_NAME', $value);
                $setting->value = $value;
                break;

            case 'DEFAULT_LANGUAGE':
                session()->put('locale', $value);
                $setting->value = $value;
                $setting->save();
                return back()->with('success', 'Default Language Updated successfully');

            case 'APP_TIMEZONE':
                $this->overWriteEnvFile('APP_TIMEZONE', $value);
                $setting->value = $value;
                break;

            case 'mail_driver':
            case 'mail_host':
            case 'mail_port':
            case 'mail_from_address':
            case 'mail_from_name':
            case 'mail_encryption':
            case 'mail_username':
            case 'mail_password':
                $this->overWriteEnvFile(strtoupper($key), $value);
                $setting->value = $value;
                break;

            default:
                $setting->value = $value;
                break;
        }

        $setting->save();
    }

    return back()->with('success', translate('Backend settings saved successfully'));
}



           

    public function sendTestMail(Request $request)
    {
        try {
            if (!$request->email) {
                return back()->with('error', 'Input your email');
            }

            $content = htmlspecialchars(prelaceScript(html_entity_decode($request->message)));
            Mail::send('mail.test', ['content' => $content], function ($message) use ($request) {
                $message->to($request->email)
                        ->subject('Test Email From Bidout')
                        ->from(Config::get('mail.from.address'), Config::get('mail.from.name'));
            });

            return back()->with('success', 'Test Mail Sent Successfully');
        } catch (\Throwable $e) {
            return back()->with('error', 'Credential Wrong!');
        }
    }

    public function overWriteEnvFile($key, $value)
    {
        $path = base_path('.env');
        if (!file_exists($path)) return;

        $value = '"' . trim($value) . '"';
        $env = file_get_contents($path);

        if (strpos($env, $key) !== false) {
            $env = preg_replace("/$key=.*/", "$key=$value", $env);
        } else {
            $env .= "\n$key=$value";
        }

        file_put_contents($path, $env);
    }

}
